<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('service_categories', function (Blueprint $table) {
            $table->id();
            $table->foreignId('parent_id')->nullable()->constrained('service_categories')->nullOnDelete();
            $table->string('name');
            $table->string('slug')->unique();
            $table->text('description')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        Schema::create('services', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete(); // provider
            $table->foreignId('service_category_id')->nullable()->constrained()->nullOnDelete();
            $table->string('title');
            $table->string('slug')->unique();
            $table->text('description')->nullable();
            $table->enum('delivery_mode', ['online', 'physical', 'hybrid']);
            $table->enum('booking_mode', ['instant', 'approval_required']);
            $table->enum('scheduling_type', ['fixed_datetime', 'request_datetime', 'no_schedule']);
            $table->unsignedInteger('duration_minutes')->nullable();
            $table->unsignedInteger('buffer_before_minutes')->default(0);
            $table->unsignedInteger('buffer_after_minutes')->default(0);
            $table->unsignedInteger('lead_time_minutes')->default(0);
            $table->string('cancellation_policy')->default('moderate');
            $table->string('currency', 3)->default('USD');
            $table->decimal('base_price', 12, 2)->default(0);
            $table->unsignedInteger('capacity')->default(1);
            $table->string('location')->nullable();
            $table->json('service_area')->nullable();
            $table->json('media')->nullable();
            $table->json('add_ons')->nullable();
            $table->unsignedInteger('approval_timeout_hours')->default(24);
            $table->unsignedInteger('auto_complete_hours')->default(24);
            $table->unsignedInteger('dispute_window_hours')->default(48);
            $table->enum('status', ['draft', 'published', 'blocked'])->default('draft');
            $table->json('meta')->nullable();
            $table->timestamps();
        });

        Schema::create('availability_rules', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete(); // provider
            $table->foreignId('service_id')->nullable()->constrained()->cascadeOnDelete();
            $table->unsignedTinyInteger('day_of_week')->nullable(); // 0-6 (sun-sat)
            $table->time('start_time')->nullable();
            $table->time('end_time')->nullable();
            $table->string('timezone')->default('UTC');
            $table->json('breaks')->nullable();
            $table->unsignedInteger('min_notice_minutes')->default(0);
            $table->unsignedInteger('max_bookings_per_day')->default(0);
            $table->unsignedInteger('buffer_before_minutes')->default(0);
            $table->unsignedInteger('buffer_after_minutes')->default(0);
            $table->boolean('is_default')->default(false);
            $table->timestamps();
        });

        Schema::create('availability_exceptions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->foreignId('service_id')->nullable()->constrained()->cascadeOnDelete();
            $table->date('date');
            $table->boolean('is_closed')->default(false);
            $table->json('slots')->nullable();
            $table->string('timezone')->default('UTC');
            $table->timestamps();
        });

        Schema::create('payment_gateways', function (Blueprint $table) {
            $table->id();
            $table->string('name')->unique(); // driver name
            $table->string('display_name');
            $table->string('mode')->default('test'); // test/live
            $table->boolean('enabled')->default(false);
            $table->json('credentials')->nullable();
            $table->json('supported_currencies')->nullable();
            $table->string('webhook_secret')->nullable();
            $table->timestamps();
        });

        Schema::create('services_calendar_tokens', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete();
            $table->string('provider'); // google
            $table->json('access_token');
            $table->string('refresh_token')->nullable();
            $table->timestamp('expires_at')->nullable();
            $table->json('scopes')->nullable();
            $table->string('remote_calendar_id')->nullable();
            $table->timestamps();
        });

        Schema::create('bookings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('service_id')->constrained()->cascadeOnDelete();
            $table->foreignId('customer_id')->constrained('users')->cascadeOnDelete();
            $table->foreignId('provider_id')->constrained('users')->cascadeOnDelete();
            $table->enum('status', [
                'draft',
                'pending_payment',
                'pending_provider_approval',
                'confirmed',
                'in_progress',
                'completed_pending_release',
                'released',
                'cancelled_by_customer',
                'cancelled_by_provider',
                'expired',
                'disputed',
                'refunded',
                'partial_refund',
                'failed_payment',
            ])->default('draft');
            $table->enum('booking_mode', ['instant', 'approval_required']);
            $table->enum('scheduling_type', ['fixed_datetime', 'request_datetime', 'no_schedule']);
            $table->enum('delivery_mode', ['online', 'physical', 'hybrid']);
            $table->dateTime('scheduled_start')->nullable();
            $table->dateTime('scheduled_end')->nullable();
            $table->string('timezone')->default('UTC');
            $table->decimal('amount', 12, 2)->default(0);
            $table->decimal('escrow_amount', 12, 2)->default(0);
            $table->string('currency', 3)->default('USD');
            $table->foreignId('payment_gateway_id')->nullable()->constrained('payment_gateways')->nullOnDelete();
            $table->json('meta')->nullable();
            $table->text('notes')->nullable();
            $table->text('provider_notes')->nullable();
            $table->dateTime('approval_expires_at')->nullable();
            $table->dateTime('auto_complete_at')->nullable();
            $table->dateTime('dispute_deadline_at')->nullable();
            $table->timestamps();
        });

        Schema::create('booking_slots', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->constrained()->cascadeOnDelete();
            $table->dateTime('start');
            $table->dateTime('end')->nullable();
            $table->enum('status', ['preferred', 'confirmed', 'proposed', 'rejected'])->default('preferred');
            $table->timestamps();
        });

        Schema::create('payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->constrained()->cascadeOnDelete();
            $table->foreignId('payment_gateway_id')->constrained('payment_gateways')->cascadeOnDelete();
            $table->string('transaction_id')->nullable();
            $table->decimal('amount', 12, 2);
            $table->string('currency', 3)->default('USD');
            $table->enum('status', ['initiated', 'authorized', 'captured', 'failed', 'refunded', 'partial_refund'])->default('initiated');
            $table->json('payload')->nullable();
            $table->timestamp('captured_at')->nullable();
            $table->timestamp('refunded_at')->nullable();
            $table->timestamps();
        });

        Schema::create('escrow_ledgers', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('payment_id')->nullable()->constrained('payments')->nullOnDelete();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->enum('type', ['deposit', 'hold', 'release', 'fee', 'refund', 'chargeback', 'payout']);
            $table->decimal('amount', 12, 2);
            $table->string('currency', 3)->default('USD');
            $table->json('metadata')->nullable();
            $table->timestamps();
        });

        Schema::create('payouts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('provider_id')->constrained('users')->cascadeOnDelete();
            $table->foreignId('booking_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('payment_gateway_id')->nullable()->constrained('payment_gateways')->nullOnDelete();
            $table->decimal('amount', 12, 2);
            $table->string('currency', 3)->default('USD');
            $table->enum('status', ['pending', 'processing', 'completed', 'failed'])->default('pending');
            $table->json('payload')->nullable();
            $table->timestamps();
        });

        Schema::create('disputes', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->constrained()->cascadeOnDelete();
            $table->foreignId('opened_by')->constrained('users')->cascadeOnDelete();
            $table->enum('status', ['open', 'under_review', 'resolved', 'rejected'])->default('open');
            $table->string('reason');
            $table->text('details')->nullable();
            $table->text('resolution')->nullable();
            $table->timestamp('opened_at')->useCurrent();
            $table->timestamp('resolved_at')->nullable();
            $table->timestamps();
        });

        Schema::create('messages', function (Blueprint $table) {
            $table->id();
            $table->foreignId('booking_id')->constrained()->cascadeOnDelete();
            $table->foreignId('sender_id')->constrained('users')->cascadeOnDelete();
            $table->foreignId('receiver_id')->constrained('users')->cascadeOnDelete();
            $table->text('body');
            $table->json('attachments')->nullable();
            $table->timestamps();
        });

        Schema::create('webhook_events', function (Blueprint $table) {
            $table->id();
            $table->foreignId('payment_gateway_id')->constrained('payment_gateways')->cascadeOnDelete();
            $table->string('event_id')->nullable();
            $table->string('type')->nullable();
            $table->json('payload');
            $table->boolean('processed')->default(false);
            $table->timestamp('processed_at')->nullable();
            $table->timestamps();
        });

        Schema::create('audit_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->morphs('auditable');
            $table->string('action');
            $table->json('metadata')->nullable();
            $table->string('ip_address')->nullable();
            $table->timestamps();
        });

        Schema::create('settings', function (Blueprint $table) {
            $table->id();
            $table->string('key')->unique();
            $table->text('value')->nullable();
            $table->timestamps();
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('audit_logs');
        Schema::dropIfExists('webhook_events');
        Schema::dropIfExists('messages');
        Schema::dropIfExists('disputes');
        Schema::dropIfExists('payouts');
        Schema::dropIfExists('escrow_ledgers');
        Schema::dropIfExists('payments');
        Schema::dropIfExists('booking_slots');
        Schema::dropIfExists('bookings');
        Schema::dropIfExists('services_calendar_tokens');
        Schema::dropIfExists('payment_gateways');
        Schema::dropIfExists('availability_exceptions');
        Schema::dropIfExists('availability_rules');
        Schema::dropIfExists('services');
        Schema::dropIfExists('service_categories');
        Schema::dropIfExists('settings');
    }
};
