<?php

namespace App\Filament\Widgets;

use App\Models\Booking;
use Carbon\CarbonPeriod;
use Filament\Widgets\ChartWidget;

class BookingTrendChart extends ChartWidget
{
    protected ?string $heading = '30d Booking Trend';

    protected static ?int $sort = 1;

    protected function getType(): string
    {
        return 'bar';
    }

    protected function getData(): array
    {
        $start = now()->subDays(29)->startOfDay();
        $period = CarbonPeriod::create($start, '1 day', now());

        $raw = Booking::selectRaw('DATE(created_at) as day, COUNT(*) as total, SUM(amount) as revenue')
            ->where('created_at', '>=', $start)
            ->groupBy('day')
            ->orderBy('day')
            ->pluck('total', 'day')
            ->toArray();

        $rawRevenue = Booking::selectRaw('DATE(created_at) as day, SUM(amount) as revenue')
            ->where('created_at', '>=', $start)
            ->groupBy('day')
            ->orderBy('day')
            ->pluck('revenue', 'day')
            ->toArray();

        $labels = [];
        $counts = [];
        $revenue = [];

        foreach ($period as $date) {
            $key = $date->format('Y-m-d');
            $labels[] = $date->format('M j');
            $counts[] = (int) ($raw[$key] ?? 0);
            $revenue[] = (float) ($rawRevenue[$key] ?? 0);
        }

        return [
            'datasets' => [
                [
                    'type' => 'line',
                    'label' => 'Bookings',
                    'data' => $counts,
                    'borderColor' => '#10b981',
                    'backgroundColor' => 'rgba(16, 185, 129, 0.2)',
                    'tension' => 0.35,
                ],
                [
                    'type' => 'bar',
                    'label' => 'Revenue',
                    'data' => $revenue,
                    'backgroundColor' => 'rgba(59, 130, 246, 0.35)',
                    'borderColor' => '#3b82f6',
                    'yAxisID' => 'y1',
                ],
            ],
            'labels' => $labels,
            'options' => [
                'scales' => [
                    'y' => ['beginAtZero' => true],
                    'y1' => [
                        'beginAtZero' => true,
                        'position' => 'right',
                        'grid' => ['display' => false],
                        'ticks' => [
                            'callback' => 'value => `$${value}`',
                        ],
                    ],
                ],
            ],
        ];
    }
}
