<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ServiceResource\Pages;
use App\Models\Service;
use Filament\Forms;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class ServiceResource extends Resource
{
    protected static ?string $model = Service::class;

    protected static \UnitEnum|string|null $navigationGroup = 'Operations';

    protected static \BackedEnum|string|null $navigationIcon = 'heroicon-o-briefcase';

    public static function form(Schema $schema): Schema
    {
        return $schema->schema([
            Forms\Components\Section::make('Basics')->schema([
                Forms\Components\TextInput::make('title')->required(),
                Forms\Components\TextInput::make('slug')->required()->unique(ignoreRecord: true),
                Forms\Components\Textarea::make('description'),
                Forms\Components\Select::make('user_id')->relationship('provider', 'name')->label('Provider')->required()->searchable(),
                Forms\Components\Select::make('service_category_id')->relationship('category', 'name')->label('Category')->searchable(),
                Forms\Components\Select::make('delivery_mode')->options([
                    'online' => 'Online',
                    'physical' => 'Physical',
                    'hybrid' => 'Hybrid',
                ])->required(),
                Forms\Components\Select::make('booking_mode')->options([
                    'instant' => 'Instant',
                    'approval_required' => 'Approval Required',
                ])->required(),
                Forms\Components\Select::make('scheduling_type')->options([
                    'fixed_datetime' => 'Fixed Datetime',
                    'request_datetime' => 'Request Datetime',
                    'no_schedule' => 'No Schedule',
                ])->required(),
                Forms\Components\TextInput::make('base_price')->numeric()->required(),
                Forms\Components\TextInput::make('currency')->default('USD')->required(),
            ])->columns(2),
            Forms\Components\Section::make('Timing')->schema([
                Forms\Components\TextInput::make('duration_minutes')->numeric()->required(),
                Forms\Components\TextInput::make('buffer_before_minutes')->numeric()->default(0),
                Forms\Components\TextInput::make('buffer_after_minutes')->numeric()->default(0),
                Forms\Components\TextInput::make('lead_time_minutes')->numeric()->default(0),
                Forms\Components\TextInput::make('approval_timeout_hours')->numeric()->default(24),
                Forms\Components\TextInput::make('auto_complete_hours')->numeric()->default(24),
                Forms\Components\TextInput::make('dispute_window_hours')->numeric()->default(48),
            ])->columns(3),
            Forms\Components\Section::make('Policies')->schema([
                Forms\Components\Select::make('cancellation_policy')->options([
                    'flexible' => 'Flexible',
                    'moderate' => 'Moderate',
                    'strict' => 'Strict',
                ])->default('moderate'),
                Forms\Components\TextInput::make('capacity')->numeric()->default(1),
                Forms\Components\Textarea::make('location'),
                Forms\Components\Textarea::make('service_area'),
                Forms\Components\Textarea::make('add_ons'),
                Forms\Components\Select::make('status')->options([
                    'draft' => 'Draft',
                    'published' => 'Published',
                    'blocked' => 'Blocked',
                ])->default('draft'),
            ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')->searchable()->sortable(),
                Tables\Columns\TextColumn::make('provider.name')->label('Provider')->sortable(),
                Tables\Columns\TextColumn::make('category.name')->label('Category'),
                Tables\Columns\TextColumn::make('status')->badge(),
                Tables\Columns\TextColumn::make('booking_mode'),
                Tables\Columns\TextColumn::make('base_price')->money(fn ($record) => $record->currency),
                Tables\Columns\TextColumn::make('created_at')->dateTime(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')->options([
                    'draft' => 'Draft',
                    'published' => 'Published',
                    'blocked' => 'Blocked',
                ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListServices::route('/'),
            'create' => Pages\CreateService::route('/create'),
            'edit' => Pages\EditService::route('/{record}/edit'),
        ];
    }
}
