<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PageResource\Pages;
use App\Models\Page;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class PageResource extends Resource
{
    protected static ?string $model = Page::class;

    protected static \UnitEnum|string|null $navigationGroup = 'CMS';

    protected static \BackedEnum|string|null $navigationIcon = 'heroicon-o-document-text';

    public static function form(Schema $schema): Schema
    {
        return $schema->schema([
            Forms\Components\Tabs::make('Page')
                ->tabs([
                    Forms\Components\Tabs\Tab::make('Content')->schema([
                        Forms\Components\TextInput::make('title')->required(),
                        Forms\Components\TextInput::make('slug')->required()->unique(ignoreRecord: true),
                        Forms\Components\Select::make('template')
                            ->options([
                                'home' => 'Home',
                                'about' => 'About',
                                'faq' => 'FAQs',
                                'custom' => 'Custom',
                            ])
                            ->default('custom')
                            ->native(false),
                        Forms\Components\Toggle::make('published')->default(true),
                        Forms\Components\RichEditor::make('content')->label('Fallback Content')->columnSpanFull(),
                    ])->columns(2),
                    Forms\Components\Tabs\Tab::make('Sections')->schema([
                        Forms\Components\Repeater::make('sections')
                            ->label('Page Sections')
                            ->orderable()
                            ->collapsed()
                            ->schema([
                                Forms\Components\Select::make('type')
                                    ->options([
                                        'hero' => 'Hero',
                                        'content' => 'Content',
                                        'features' => 'Features',
                                        'faq' => 'FAQs',
                                        'stats' => 'Stats',
                                        'cta' => 'Call to Action',
                                    ])
                                    ->required()
                                    ->native(false),
                                Forms\Components\TextInput::make('title')->label('Heading')->required(),
                                Forms\Components\TextInput::make('subtitle')->label('Subheading'),
                                Forms\Components\Textarea::make('body')->rows(3),
                                Forms\Components\TextInput::make('cta_label')->label('CTA Label'),
                                Forms\Components\TextInput::make('cta_url')->label('CTA URL'),
                                Forms\Components\Repeater::make('items')
                                    ->label('Items')
                                    ->schema([
                                        Forms\Components\TextInput::make('title')->label('Label')->required(),
                                        Forms\Components\Textarea::make('description')->rows(2),
                                    ])
                                    ->default([])
                                    ->collapsed()
                                    ->columns(2)
                                    ->visible(fn (Forms\Get $get) => in_array($get('type'), ['features', 'faq', 'stats'])),
                            ])
                            ->columnSpanFull(),
                    ]),
                    Forms\Components\Tabs\Tab::make('SEO')->schema([
                        Forms\Components\TextInput::make('seo.title')->label('Meta Title'),
                        Forms\Components\Textarea::make('seo.description')->label('Meta Description')->rows(2),
                        Forms\Components\TextInput::make('seo.keywords')->helperText('Comma separated'),
                    ])->columns(2),
                ]),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')->searchable(),
                Tables\Columns\TextColumn::make('slug')->searchable(),
                Tables\Columns\TextColumn::make('template')
                    ->badge()
                    ->color(fn ($state) => match ($state) {
                        'home' => 'success',
                        'about' => 'warning',
                        'faq' => 'info',
                        default => 'primary',
                    }),
                Tables\Columns\IconColumn::make('published')->boolean(),
                Tables\Columns\TextColumn::make('created_at')->dateTime(),
                Tables\Columns\TextColumn::make('updated_at')->since(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPages::route('/'),
            'create' => Pages\CreatePage::route('/create'),
            'edit' => Pages\EditPage::route('/{record}/edit'),
        ];
    }
}
