<?php

namespace App\Filament\Resources;

use App\Filament\Resources\BookingResource\Pages;
use App\Models\Booking;
use Filament\Forms;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;

class BookingResource extends Resource
{
    protected static ?string $model = Booking::class;

    protected static \UnitEnum|string|null $navigationGroup = 'Operations';

    protected static \BackedEnum|string|null $navigationIcon = 'heroicon-o-calendar';

    public static function form(Schema $schema): Schema
    {
        return $schema->schema([
            Forms\Components\Select::make('service_id')->relationship('service', 'title')->searchable()->required(),
            Forms\Components\Select::make('customer_id')->relationship('customer', 'name')->required(),
            Forms\Components\Select::make('provider_id')->relationship('provider', 'name')->required(),
            Forms\Components\Select::make('status')->options([
                'draft' => 'draft',
                'pending_payment' => 'pending_payment',
                'pending_provider_approval' => 'pending_provider_approval',
                'confirmed' => 'confirmed',
                'in_progress' => 'in_progress',
                'completed_pending_release' => 'completed_pending_release',
                'released' => 'released',
                'cancelled_by_customer' => 'cancelled_by_customer',
                'cancelled_by_provider' => 'cancelled_by_provider',
                'expired' => 'expired',
                'disputed' => 'disputed',
                'refunded' => 'refunded',
                'partial_refund' => 'partial_refund',
                'failed_payment' => 'failed_payment',
            ])->required(),
            Forms\Components\DateTimePicker::make('scheduled_start'),
            Forms\Components\DateTimePicker::make('scheduled_end'),
            Forms\Components\TextInput::make('amount')->numeric()->required(),
            Forms\Components\TextInput::make('currency')->default('USD'),
            Forms\Components\Textarea::make('notes'),
            Forms\Components\Textarea::make('provider_notes'),
        ])->columns(2);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')->sortable(),
                Tables\Columns\TextColumn::make('service.title')->label('Service')->searchable(),
                Tables\Columns\TextColumn::make('customer.name')->label('Customer'),
                Tables\Columns\TextColumn::make('provider.name')->label('Provider'),
                Tables\Columns\TextColumn::make('status')->badge(),
                Tables\Columns\TextColumn::make('scheduled_start')->dateTime(),
                Tables\Columns\TextColumn::make('amount')->money(fn ($record) => $record->currency),
                Tables\Columns\TextColumn::make('created_at')->dateTime(),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')->options([
                    'pending_payment' => 'pending_payment',
                    'pending_provider_approval' => 'pending_provider_approval',
                    'confirmed' => 'confirmed',
                    'disputed' => 'disputed',
                ]),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBookings::route('/'),
            'create' => Pages\CreateBooking::route('/create'),
            'edit' => Pages\EditBooking::route('/{record}/edit'),
        ];
    }
}
